<?php
/**
 * Web-based Cookie Converter for ChatGPT Proxy
 * Allows uploading localStorage.json and cookies.json files from local computer
 */

class UploadCookieConverter {
    private $config;
    private $basePath;
    private $accountPath;
    private $uploadDir;
    
    public function __construct() {
        // Load configuration
        $configFile = __DIR__ . '/config.json';
        if (!file_exists($configFile)) {
            $this->config = ['accounts' => ['account_1']]; // Default config
        } else {
            $this->config = json_decode(file_get_contents($configFile), true);
            if (!$this->config) {
                $this->config = ['accounts' => ['account_1']]; // Default config
            }
        }
        
        $this->basePath = __DIR__ . '/';
        $this->accountPath = $this->basePath . 'account_1/';
        $this->uploadDir = $this->basePath . 'uploads/';
        
        // Ensure directories exist
        if (!is_dir($this->accountPath)) {
            mkdir($this->accountPath, 0755, true);
        }
        if (!is_dir($this->uploadDir)) {
            mkdir($this->uploadDir, 0755, true);
        }
    }
    
    public function handleUpload() {
        $result = ['success' => false, 'message' => '', 'details' => []];
        
        try {
            // Check if files were uploaded
            if (!isset($_FILES['cookies_file']) || !isset($_FILES['localstorage_file'])) {
                throw new Exception('Both cookies.json and localStorage.json files are required');
            }
            
            $cookiesFile = $_FILES['cookies_file'];
            $localStorageFile = $_FILES['localstorage_file'];
            
            // Validate file uploads
            if ($cookiesFile['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('Error uploading cookies.json: ' . $this->getUploadError($cookiesFile['error']));
            }
            
            if ($localStorageFile['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('Error uploading localStorage.json: ' . $this->getUploadError($localStorageFile['error']));
            }
            
            // Validate file types
            if (!$this->isJsonFile($cookiesFile)) {
                throw new Exception('cookies.json must be a valid JSON file');
            }
            
            if (!$this->isJsonFile($localStorageFile)) {
                throw new Exception('localStorage.json must be a valid JSON file');
            }
            
            // Save uploaded files temporarily
            $cookiesPath = $this->uploadDir . 'cookies_' . time() . '.json';
            $localStoragePath = $this->uploadDir . 'localStorage_' . time() . '.json';
            
            if (!move_uploaded_file($cookiesFile['tmp_name'], $cookiesPath)) {
                throw new Exception('Failed to save cookies.json file');
            }
            
            if (!move_uploaded_file($localStorageFile['tmp_name'], $localStoragePath)) {
                throw new Exception('Failed to save localStorage.json file');
            }
            
            // Process the files
            $conversionResult = $this->convertFiles($cookiesPath, $localStoragePath);
            
            // Clean up temporary files
            unlink($cookiesPath);
            unlink($localStoragePath);
            
            $result['success'] = true;
            $result['message'] = 'Files uploaded and converted successfully!';
            $result['details'] = $conversionResult;
            
        } catch (Exception $e) {
            $result['message'] = $e->getMessage();
        }
        
        return $result;
    }
    
    private function convertFiles($cookiesPath, $localStoragePath) {
        $details = [];
        
        // Read and validate cookies file
        $cookiesData = file_get_contents($cookiesPath);
        $chromeCookies = json_decode($cookiesData, true);
        
        if (!$chromeCookies) {
            throw new Exception('Invalid cookies.json format');
        }
        
        // Read and validate localStorage file
        $localStorageData = file_get_contents($localStoragePath);
        $localStorageJson = json_decode($localStorageData, true);
        
        if ($localStorageJson === null) {
            throw new Exception('Invalid localStorage.json format');
        }
        
        $details[] = "Found " . count($chromeCookies) . " cookies to convert";
        $details[] = "localStorage.json contains " . count($localStorageJson) . " entries";
        
        // Convert cookies to proxy format
        $this->convertToProxyFormat($chromeCookies, $details);
        
        // Save localStorage.json
        file_put_contents($this->accountPath . 'localStorage.json', $localStorageData);
        $details[] = "✓ Saved localStorage.json with uploaded data";
        
        // Create other required files
        $this->createRequiredFiles($details);
        
        return $details;
    }
    
    private function convertToProxyFormat($chromeCookies, &$details) {
        $proxyJsonCookies = [];
        $proxyCookiesArray = [];
        
        foreach ($chromeCookies as $cookie) {
            // Skip if missing required fields
            if (!isset($cookie['name']) || !isset($cookie['value'])) {
                continue;
            }
            
            // Convert Chrome format to proxy JSON format (compact like working version)
            $proxyJsonCookie = [
                'name' => $cookie['name'],
                'value' => $cookie['value'],
                'domain' => $cookie['domain'] ?? '.chatgpt.com',
                'path' => $cookie['path'] ?? '/',
                'httpOnly' => $cookie['httpOnly'] ?? false,
                'secure' => $cookie['secure'] ?? false,
                'sameSite' => $this->convertSameSiteWorking($cookie['sameSite'] ?? 'lax')
            ];
            
            // Handle expiration - use 'expires' field like working version
            if (isset($cookie['expirationDate']) && $cookie['expirationDate'] > 0) {
                $proxyJsonCookie['expires'] = $cookie['expirationDate'];
            } elseif (isset($cookie['expires']) && $cookie['expires'] > 0) {
                $proxyJsonCookie['expires'] = $cookie['expires'];
            } else {
                $proxyJsonCookie['expires'] = -1; // Session cookie
            }
            
            $proxyJsonCookies[] = $proxyJsonCookie;
            
            // Convert to cookies.file format (matching working version exactly)
            $cookieData = [
                'value' => $cookie['value'] . '; expires', // Add expires suffix like working version
                'domain' => $cookie['domain'] ?? '.chatgpt.com',
                'path' => $cookie['path'] ?? '/',
                'samesite' => $this->convertSameSiteForFile($cookie['sameSite'] ?? 'lax'),
                'max-age' => '0' // Add max-age like working version
            ];
            
            // Add expires if available
            if (isset($cookie['expirationDate']) && $cookie['expirationDate'] > 0) {
                $cookieData['expires'] = date('D, d-M-y H:i:s T', (int)$cookie['expirationDate']);
            } elseif (isset($cookie['expires']) && $cookie['expires'] > 0) {
                $cookieData['expires'] = date('D, d-M-y H:i:s T', (int)$cookie['expires']);
            } else {
                $cookieData['expires'] = 'Wed, 31-Dec-1969 23:59:59 GMT'; // Default for session cookies
            }
            
            $proxyCookiesArray[$cookie['name']] = $cookieData;
        }
        
        // Write cookies.json (compact format like working version)
        $jsonOutput = json_encode($proxyJsonCookies, JSON_UNESCAPED_SLASHES);
        file_put_contents($this->accountPath . 'cookies.json', $jsonOutput);
        $details[] = "✓ Created cookies.json (" . count($proxyJsonCookies) . " cookies)";
        
        // Write cookies.file (PHP serialized format)
        $serializedOutput = serialize($proxyCookiesArray);
        file_put_contents($this->accountPath . 'cookies.file', $serializedOutput);
        $details[] = "✓ Created cookies.file (" . count($proxyCookiesArray) . " cookies)";
    }
    
    private function convertSameSiteWorking($chromeSameSite) {
        $mapping = [
            'no_restriction' => 'None',
            'None' => 'None',
            'lax' => 'Lax',
            'Lax' => 'Lax',
            'strict' => 'Strict',
            'Strict' => 'Strict'
        ];
        
        return $mapping[strtolower($chromeSameSite)] ?? 'Lax';
    }
    
    private function convertSameSiteForFile($chromeSameSite) {
        $mapping = [
            'no_restriction' => 'None',
            'None' => 'None',
            'lax' => 'Lax',
            'Lax' => 'Lax',
            'strict' => 'Strict',
            'Strict' => 'Strict'
        ];
        
        return $mapping[strtolower($chromeSameSite)] ?? 'Lax';
    }
    
    private function createRequiredFiles(&$details) {
        // Create auth.json if it doesn't exist
        $authFile = $this->accountPath . 'auth.json';
        if (!file_exists($authFile)) {
            $authData = [
                'access_token' => '',
                'refresh_token' => '',
                'expires_in' => 0,
                'token_type' => 'Bearer'
            ];
            file_put_contents($authFile, json_encode($authData, JSON_PRETTY_PRINT));
            $details[] = "✓ Created auth.json template";
        }
        
        // Create auth.required if it doesn't exist
        $authRequiredFile = $this->accountPath . 'auth.required';
        if (!file_exists($authRequiredFile)) {
            file_put_contents($authRequiredFile, '');
            $details[] = "✓ Created auth.required";
        }
        
        // Create ua.txt if it doesn't exist
        $uaFile = $this->accountPath . 'ua.txt';
        if (!file_exists($uaFile)) {
            $userAgent = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36';
            file_put_contents($uaFile, $userAgent);
            $details[] = "✓ Created ua.txt";
        }
        
        // Create main.html if it doesn't exist
        $mainHtmlFile = $this->accountPath . 'main.html';
        if (!file_exists($mainHtmlFile)) {
            $html = '<!DOCTYPE html><html><head><title>ChatGPT Proxy</title></head><body><h1>ChatGPT Proxy Account</h1></body></html>';
            file_put_contents($mainHtmlFile, $html);
            $details[] = "✓ Created main.html";
        }
    }
    
    private function isJsonFile($file) {
        // Check file extension
        $filename = $file['name'];
        if (!preg_match('/\.json$/i', $filename)) {
            return false;
        }
        
        // Try to read and parse JSON
        $content = file_get_contents($file['tmp_name']);
        $json = json_decode($content, true);
        
        return $json !== null;
    }
    
    private function getUploadError($errorCode) {
        switch ($errorCode) {
            case UPLOAD_ERR_INI_SIZE:
                return 'File too large (exceeds upload_max_filesize)';
            case UPLOAD_ERR_FORM_SIZE:
                return 'File too large (exceeds MAX_FILE_SIZE)';
            case UPLOAD_ERR_PARTIAL:
                return 'File was only partially uploaded';
            case UPLOAD_ERR_NO_FILE:
                return 'No file was uploaded';
            case UPLOAD_ERR_NO_TMP_DIR:
                return 'Missing temporary folder';
            case UPLOAD_ERR_CANT_WRITE:
                return 'Failed to write file to disk';
            case UPLOAD_ERR_EXTENSION:
                return 'Upload stopped by PHP extension';
            default:
                return 'Unknown upload error';
        }
    }
}

// Handle POST request (file upload)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
    
    $converter = new UploadCookieConverter();
    $result = $converter->handleUpload();
    
    echo json_encode($result);
    exit;
}

// HTML interface
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>ChatGPT Proxy - Cookie Converter</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .container {
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            padding: 40px;
            max-width: 600px;
            width: 100%;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .header h1 {
            color: #333;
            font-size: 2.5em;
            margin-bottom: 10px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .header p {
            color: #666;
            font-size: 1.1em;
        }
        
        .upload-form {
            margin-bottom: 30px;
        }
        
        .file-group {
            margin-bottom: 25px;
        }
        
        .file-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #333;
            font-size: 1.1em;
        }
        
        .file-input-wrapper {
            position: relative;
            display: inline-block;
            width: 100%;
        }
        
        .file-input {
            width: 100%;
            padding: 15px;
            border: 2px dashed #ddd;
            border-radius: 10px;
            background: #f9f9f9;
            cursor: pointer;
            transition: all 0.3s ease;
            font-size: 1em;
        }
        
        .file-input:hover {
            border-color: #667eea;
            background: #f0f4ff;
        }
        
        .file-input:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .submit-btn {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 1.1em;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .submit-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(102, 126, 234, 0.3);
        }
        
        .submit-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .result {
            margin-top: 20px;
            padding: 20px;
            border-radius: 10px;
            display: none;
        }
        
        .result.success {
            background: #d4edda;
            border: 1px solid #c3e6cb;
            color: #155724;
        }
        
        .result.error {
            background: #f8d7da;
            border: 1px solid #f5c6cb;
            color: #721c24;
        }
        
        .result h3 {
            margin-bottom: 10px;
            font-size: 1.2em;
        }
        
        .result ul {
            list-style: none;
            padding-left: 0;
        }
        
        .result li {
            padding: 5px 0;
            border-bottom: 1px solid rgba(0,0,0,0.1);
        }
        
        .result li:last-child {
            border-bottom: none;
        }
        
        .instructions {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-top: 30px;
        }
        
        .instructions h3 {
            color: #333;
            margin-bottom: 15px;
            font-size: 1.3em;
        }
        
        .instructions ol {
            color: #666;
            line-height: 1.6;
        }
        
        .instructions li {
            margin-bottom: 8px;
        }
        
        .loading {
            display: none;
            text-align: center;
            margin-top: 20px;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🍪 Cookie Converter</h1>
            <p>Upload your browser session files to convert for ChatGPT Proxy</p>
        </div>
        
        <form id="uploadForm" class="upload-form" enctype="multipart/form-data">
            <div class="file-group">
                <label for="cookies_file">📁 cookies.json (Chrome Extension Export)</label>
                <input type="file" id="cookies_file" name="cookies_file" accept=".json" class="file-input" required>
            </div>
            
            <div class="file-group">
                <label for="localstorage_file">💾 localStorage.json (Browser Storage)</label>
                <input type="file" id="localstorage_file" name="localstorage_file" accept=".json" class="file-input" required>
            </div>
            
            <button type="submit" class="submit-btn">🚀 Convert Files</button>
        </form>
        
        <div class="loading" id="loading">
            <div class="spinner"></div>
            <p>Converting files...</p>
        </div>
        
        <div id="result" class="result"></div>
        
        <div class="instructions">
            <h3>📋 Instructions</h3>
            <ol>
                <li><strong>Export cookies.json:</strong> Use a Chrome extension like "cookies.txt" to export ChatGPT cookies in JSON format</li>
                <li><strong>Get localStorage.json:</strong> Use browser developer tools (F12) → Application → Local Storage → chatgpt.com, then export the data</li>
                <li><strong>Upload both files:</strong> Select the files using the form above</li>
                <li><strong>Convert:</strong> Click "Convert Files" to process and create proxy-compatible files</li>
                <li><strong>Files created:</strong> The converter will create all required files in the account_1/ folder</li>
            </ol>
        </div>
    </div>

    <script>
        document.getElementById('uploadForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const formData = new FormData(this);
            const submitBtn = document.querySelector('.submit-btn');
            const loading = document.getElementById('loading');
            const result = document.getElementById('result');
            
            // Show loading
            submitBtn.disabled = true;
            loading.style.display = 'block';
            result.style.display = 'none';
            
            try {
                const response = await fetch('', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                // Hide loading
                loading.style.display = 'none';
                submitBtn.disabled = false;
                
                // Show result
                result.style.display = 'block';
                result.className = 'result ' + (data.success ? 'success' : 'error');
                
                if (data.success) {
                    result.innerHTML = `
                        <h3>✅ Conversion Successful!</h3>
                        <p>${data.message}</p>
                        <ul>
                            ${data.details.map(detail => `<li>${detail}</li>`).join('')}
                        </ul>
                    `;
                } else {
                    result.innerHTML = `
                        <h3>❌ Conversion Failed</h3>
                        <p>${data.message}</p>
                    `;
                }
                
            } catch (error) {
                loading.style.display = 'none';
                submitBtn.disabled = false;
                
                result.style.display = 'block';
                result.className = 'result error';
                result.innerHTML = `
                    <h3>❌ Upload Error</h3>
                    <p>Failed to upload files: ${error.message}</p>
                `;
            }
        });
        
        // File input change handlers for better UX
        document.querySelectorAll('.file-input').forEach(input => {
            input.addEventListener('change', function() {
                if (this.files.length > 0) {
                    this.style.borderColor = '#28a745';
                    this.style.background = '#f0fff4';
                }
            });
        });
    </script>
</body>
</html>
